package org.msh.tb.bd.tbforms.dhis2;

import org.jboss.seam.Component;
import org.jboss.seam.annotations.In;
import org.jboss.seam.contexts.Contexts;
import org.jboss.seam.security.Identity;
import org.msh.tb.application.EtbmanagerApp;
import org.msh.tb.application.tasks.TaskManager;
import org.msh.tb.bd.tbforms.dhis2.tb10.DHIS2ExportTB10AsyncTask;
import org.msh.tb.application.tasks.AsyncTask;
import org.msh.tb.entities.TaskLog;
import org.msh.tb.entities.Tbunit;
import org.msh.tb.entities.UserWorkspace;
import org.msh.tb.entities.Workspace;
import org.msh.tb.indicators.core.IndicatorFilters;
import org.msh.tb.login.UserSession;
import org.msh.tb.tbunits.TBUnitSelection2;
import org.msh.utils.date.DateUtils;

import javax.persistence.EntityManager;
import javax.persistence.Query;
import java.util.*;

/**
 * Created by Mauricio on 15/02/2017.
 * Abstract class with common code used on all TB Form exportation
 */
public abstract class DHIS2ExportHome {

    @In(create=true)
    protected EntityManager entityManager;

    @In(create=true)
    protected DHIS2AuthenticationService dHIS2AuthenticationService;

    @In(create=true)
    protected DHIS2DataExportService dHIS2DataExportService;

    @In
    protected TaskManager taskManager;

    @In(create=true)
    EtbmanagerApp etbmanagerApp;

    /**
     * Stores task id during request to be used bu getStatus method
     */
    private String taskId;

    /**
     * Get values selected by the user on filters
     */
    private IndicatorFilters indicatorFilter;

    /**
     * Stores units list during request
     */
    protected List<Tbunit> units;

    /**
     * Starts export process in management module.
     * As this method considers all tbunits selected inside tbunitselection, only users that can play other unit are allowed to access this method.
     * Mounts params set passed to task containing selected units, selected quarter and request services.
     */
    public void export(){
        if (getTask() != null) {
            throw new RuntimeException("The export task is already running");
        }

        if (!Identity.instance().hasRole(getReportRole())) {
            throw new RuntimeException("You don't have permission to do this");
        }

        // todo: maybe it should be removed, added this to avoid sending data about a wrong quarter during tests
        if(etbmanagerApp.getConfiguration().getIntegrateDHIS2After().after(getIndicatorFilters().getQuarter().getIniDate())) {
            throw new RuntimeException("You can't send data about that quarter");
        }

        Map<String, Object> params =  new HashMap<String, Object>();

        List<Tbunit> tbunits = loadTbUnitsList();

        // check if user can export units in list
        if (UserSession.isTbFormUnitView()) {
            for (Tbunit u : tbunits) {
                if (!u.getId().equals(UserSession.getUserWorkspace().getTbunit().getId())){
                    throw new RuntimeException("You are not allowed to export a tbunit on the list.");
                }
            }
        }

        // set params
        params.put(DHIS2ExportTB10AsyncTask.PARAM_TBUNITS, tbunits);
        params.put(DHIS2ExportTB10AsyncTask.PARAM_SELECTED_QUARTER, getIndicatorFilters().getQuarter());
        params.put(DHIS2ExportTB10AsyncTask.PARAM_DHIS2_AUTH_SERVICE, dHIS2AuthenticationService);
        params.put(DHIS2ExportTB10AsyncTask.PARAM_DHIS2_EXPORT_SERVICE, dHIS2DataExportService);

        startTask(params);
    }

    /**
     * Starts task using task manager
     * @param params
     */
    protected abstract void startTask(Map<String, Object> params);

    /**
     * Each user should run only one Tb Form Exportation at a time.
     * This method will return the task if it is running or null if it is not running.
     * @return
     */
    protected abstract AsyncTask getTask();

    /**
     * Returns the dhis2 data set id correponding to the form implementation
     * @return
     */
    public abstract String getFormId();

    /**
     * Returns the user role necessary to access this feature
     * @return
     */
    public abstract String getReportRole();

    /**
     * Returns the progress of the running Async Task
     * @return
     */
    public Integer getProgress() {
        AsyncTask task = getTask();

        // task is not running
        if (task == null) {
            return null;
        }

        return new Double(task.getProgress()).intValue();
    }

    /**
     * Returns the status of the task.
     * @return
     */
    public ExportStatus getStatus() {
        // find task using getTask method to guarantee that the same user will run the same tb form export once a time.
        AsyncTask task = getTask();

        if (task != null) {
            return ExportStatus.convert(task.getStatus());
        }

        // If has task id = task finished, error or canceled
        // If not = No task is running
        if (taskId != null && !taskId.equals("undefined") ) {
            TaskLog log = taskManager.findTaskLog(taskId);
            if (log != null) {
                return ExportStatus.convert(log.getStatus());
            }
        }

        return ExportStatus.NOT_RUNNING;
    }

    /**
     * Returns the ID of the running task
     * @return
     */
    public String findTaskId() {
        // find task using getTask method to guarantee that the same user will run the same tb form export once a time.
        AsyncTask task = getTask();

        if (task == null) {
            return null;
        }

        return task.getId();
    }

    /**
     * Return the quantity of units available for integration based on current filters.
     * @return
     */
    public Integer getUnitsCount() {
        if (units == null) {
            units = loadTbUnitsList();
        }

        return units != null ? units.size() : 0;
    }

    /**
     * Get the list of tbunit with dhis2 id informed and that matches tbunitselection condition.
     * @return
     */
    protected List<Tbunit> loadTbUnitsList() {
        List<Tbunit> ret = null;
        TBUnitSelection2 tbUnitSelection2 = getIndicatorFilters().getTbunitselection();

        if (tbUnitSelection2.getSelected() != null) {
            // Tbunit is selected, so should send only this unit.
            // check if this unit has DHIS2 ID.
            if (tbUnitSelection2.getSelected().getDhis2Id() != null && tbUnitSelection2.getSelected().isDhis2Enabled()) {
                ret = new ArrayList<Tbunit>();
                ret.add(tbUnitSelection2.getSelected());
            }
        } else {
            // Admin unit is selected or the whole workspace is selected
            String qryStr = "from Tbunit u where u.dhis2Id is not null and u.dhis2Enabled = true and u.workspace.id = :workspaceId";
            qryStr += tbUnitSelection2.getAdminUnit() != null ? " and u.adminUnit.code like :code" : "";

            Query query = entityManager.createQuery(qryStr);

            query.setParameter("workspaceId", getWorkspace().getId());

            if (tbUnitSelection2.getAdminUnit() != null) {
                query.setParameter("code", tbUnitSelection2.getAdminUnitCodeLike());
            }

            ret = query.getResultList();
        }

        return ret;
    }

    public boolean canSendData() {
        if (getIndicatorFilters().getQuarter() == null || getIndicatorFilters().getQuarter().getIniDate() == null) {
            return false;
        }
        return !etbmanagerApp.getConfiguration().getIntegrateDHIS2After().after(getIndicatorFilters().getQuarter().getIniDate());
    }

    /**
     * returns the UserWorkspace logged in
     * @return
     */
    protected UserWorkspace getUserWorkspace() {
        return ((UserWorkspace) Component.getInstance("userWorkspace", true));
    }

    /**
     * Returns the workspace that id being used
     * @return
     */
    protected Workspace getWorkspace() {
        return (Workspace) Component.getInstance("defaultWorkspace");
    }

    /**
     * Return the filters selected by the user
     * @return instance of the {@link IndicatorFilters IndicatorFilters} class
     */
    protected IndicatorFilters getIndicatorFilters() {
        if (indicatorFilter == null)
            indicatorFilter = (IndicatorFilters)Contexts.getSessionContext().get("indicatorFilters");
        return indicatorFilter;
    }

    public void setTaskId(String taskId) {
        this.taskId = taskId;
    }

    public String getTaskId() {
        return taskId;
    }
}
